/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.security.action;

import java.security.*;

/**
 * A convenience class for retrieving the boolean value of a system property
 * as a privileged action.
 *
 * @see java.security.PrivilegedAction
 * @see java.security.AccessController
 */

public class GetBooleanAction implements PrivilegedAction {
    private String prop;
    private boolean defaultVal;
    private boolean defaultSet = false;

    /**
     * Constructor that takes the name of the system property whose boolean
     * value needs to be determined.
     *
     * @param prop the name of the system property
     */
    public GetBooleanAction(String prop) {
        this.prop = prop;
    }

    /**
     * Constructor that takes the name of the system property whose boolean
     * value needs to be determined and a default value to use when property
     * is not set.
     *
     * @param prop the name of the system property
     * @param defaultVal default value
     */
    public GetBooleanAction(String prop, boolean defaultVal) {
        this.prop = prop;
        this.defaultVal = defaultVal;
        this.defaultSet = true;
    }

    /**
     * Determines the boolean value of the system property whose name was
     * specified in the constructor.
     *
     * @return the <code>Boolean</code> value of the property.
     */
    public Object run() {
        String value = System.getProperty(prop);
        if (value != null) return Boolean.valueOf(value);
        if (defaultSet) return Boolean.valueOf(defaultVal);
        return null;
    }
}
